//////////////////////////////////////////////////////////////////////////////
//  Copyright 2014 by Autodesk, Inc.  All rights reserved.
//
// This computer source code and related instructions and comments 
// are the unpublished confidential and proprietary information of 
// Autodesk, Inc. and are protected under applicable copyright and 
// trade secret law.  They may not be disclosed to, copied or used 
// by any third party without the prior written consent of Autodesk, Inc.
//////////////////////////////////////////////////////////////////////////////

/*globals $,jQuery,document, window, navigator*/

/*!
 * ntp.page - Implements animations and functionality of new tab page elements
 */

 // Namespace
var ntp = ntp || {};
ntp.page = ntp.page || {};

ntp.page.animationSettings = {
    TAB_SWITCHING_ANIMATION_DURATION: 600,
    ELEMENT_TOGGLING_ANIMATION_DURATION: 200
};

/**
 * Handles page layout that cannot be done via CSS
 */
ntp.page.layout = (function () {
    var $allTabs = null;
    var $columns = null;
    var $getStartedVideo = null;
    var $featuredVideo = null;
    var $onlineResource = null;
    var $continueWorking = null;
    var $connectPanel = null;
    var columnMinHeight = 504;
    //This is video item height. thumb height 90px, thumb border is 2px and thumb bottom-margin is 8px.
    var videoItemHeight = 100;
    //This is video item margin. video item margin-bottom is 16px.
    var videoItemGap = 16;
    
   
    var getColumnHeight = function() {
        var newHeight = $allTabs.height() * 0.7;
        if (newHeight < columnMinHeight){
            newHeight = columnMinHeight;
        }
        return newHeight;
    };
    
    var adjustTopSpace = function() {
        var newColumnHeight = $allTabs.height() * 0.7;
        var space = $allTabs.height() - newColumnHeight;
        if (space < 0)
            space = 0;
        $allTabs.children('.content').css('margin-top', 0.375 * space);
    };
    
    return {
        init: function () {
            $canvas = $('#canvas');
            $allTabs = $('#canvas_slider > .tab');
            $columns = $('section.column');
            $getStartedVideo = $('#getting_started ol');
            $featuredVideo = $('#feature_videos ol');
            $continueWorking = $('#continue_working');
            $onlineResource = $('#online_resources_panel');
            $connectPanel = $('#connect_scroll_panel');
            this.adjustColumnHeight();
        },
        
        adjustColumnHeight: function () {
            var newHeight = getColumnHeight();
            $columns.height(newHeight);
            $continueWorking.height(newHeight);
            this.updateVideoSectionHeight();
            this.updateOnlineResourceHeight();
            this.updateConnectHeight();
        },
        
        update: function () {
            // Update tab width
            var tabWidth = $canvas.width();
            $allTabs.each(function () {
                $(this).width(tabWidth);
            });
            
            this.adjustColumnHeight();
            adjustTopSpace();
        },
        
        updateVideoSectionHeight: function() {
            var height = getColumnHeight();
            if(ntp.content.learnTab.hasGetStartedVideos() && ntp.content.learnTab.hasFeaturedVideos()) {
                var getStartedItemsHeight = $getStartedVideo.children().length * videoItemHeight
                                                + ($getStartedVideo.children().length - 1) * videoItemGap;
                var featuredItemsHeight = $featuredVideo.children().length * videoItemHeight
                                                + ($featuredVideo.children().length - 1) * videoItemGap;
                var getStartedHeight = height * 0.5;
                if (getStartedHeight > getStartedItemsHeight)
                    getStartedHeight = getStartedItemsHeight;
                var featuredHeight = height - getStartedHeight
                                        - $('#getting_started_title').outerHeight(true)
                                        - $('#video_section_spacing').outerHeight(true)
                                        - $('#featured_title').outerHeight(true);
                if (featuredHeight > featuredItemsHeight){
                    featuredHeight = featuredItemsHeight;
                    getStartedHeight = height - featuredHeight
                                        - $('#getting_started_title').outerHeight(true)
                                        - $('#video_section_spacing').outerHeight(true)
                                        - $('#featured_title').outerHeight(true);
                }
                $getStartedVideo.css({'height': 'auto', 'max-height': getStartedHeight});
                $featuredVideo.css({'height': 'auto','max-height': featuredHeight});
            }
            else if(ntp.content.learnTab.hasGetStartedVideos()) {
                var getStartedHeight = height - $('#getting_started_title').outerHeight(true);
                $getStartedVideo.css({'height': 'auto','max-height': getStartedHeight});
            }
            else if(ntp.content.learnTab.hasFeaturedVideos()) {
                var featuredHeight = height - $('#featured_title').outerHeight(true);
                $featuredVideo.css({'height': 'auto', 'max-height': featuredHeight});
            }
        },
        
        updateOnlineResourceHeight: function() {
            if(ntp.content.learnTab.hasOnlineResource()) {
                var height = getColumnHeight();
                var tipHeight = 0;
                if(ntp.content.learnTab.hasTips()) {
                    tipHeight = $('#Section_tip').outerHeight(true);
                }
                var onlineHeight = height - $('#online_resource_title').outerHeight(true) - tipHeight;
                $onlineResource.css('max-height', onlineHeight);
            }
        },
        
        updateConnectHeight: function() {
            if(ntp.widget.connect.hasConnect()){
                var height = getColumnHeight();
                var panelHeight;
                if(ntp.widget.notifications.hasNotifications()){
                    panelHeight = height - $('#connect_title').outerHeight(true) - $('#section_notifications').outerHeight(true);
                }
                else {
                    panelHeight = height - $('#connect_title').outerHeight(true);
                }
                $connectPanel.css('max-height', panelHeight);
            }
        }
    };
})();

/**
 * Handles tabs and tab switching
 */
ntp.page.tabs = (function () {
    
    var $canvas = null;
    var $tabs = null;
    var tipContentsVisited = false;
    var $visableTabs = null;
    var selectedIndex = 0;
    
    return {
        current: function () {
            return selectedIndex;
        },

        isTipContentsVisited: function () {
            return tipContentsVisited;
        },
        
        count: function () {
            if ($tabs) {
                return $tabs.length;
            } else {
                return 0;
            }
        },
        
        visibleCount: function () {
            if ($visibleTabs)
                return $visibleTabs.length;
            else
                return 1;
        },
        
        init: function () {
            $canvas = $('#canvas');
            $tabs = $('#canvas_slider > .tab');
            $visibleTabs = $('#canvas_slider > .tab.show');
        },
        
        addVisibleTab: function (id, select) {
            var index = 1;
            var $tab = $visibleTabs.eq(selectedIndex - 1);
            var overCurrent = false;
            for(var i = 0; i < $tabs.length; i++){
                if($tabs.eq(i).attr('id') === id){
                    $tabs.eq(i).addClass('show');
                    $('#footer_nav li').eq(i).addClass('show');
                    $visibleTabs = $('#canvas_slider > .tab.show');
                    if(!overCurrent){
                        selectedIndex++;
                        if(!select)
                            this.update();
                    }
                    ntp.page.footerNavs.visibleTabsChange();
                    ntp.page.sideNavs.visibleTabsChange();
                    break;
                }
                else if($tabs.eq(i).attr('id') === $tab.attr('id')){
                    overCurrent = true;
                }
            }
            if(select)
                this.selectById(id, true, true);
        },
        
        removeVisibleTab: function (id, defaultId) {
            $visibleTabs = $('#canvas_slider > .tab.show');
            var $tab = $visibleTabs.eq(selectedIndex - 1);
            if ($tab.attr('id') === id) {
                $tab.removeClass('show');
                $('#footer_nav li').eq(selectedIndex - 1).removeClass('show');
                $visibleTabs = $('#canvas_slider > .tab.show');
                this.selectById(defaultId, true, true);
                ntp.page.footerNavs.visibleTabsChange();
                ntp.page.sideNavs.visibleTabsChange();
            }
            else {
                var overCurrent = false;
                for (var i = 0; i < $visibleTabs.length; i++) {
                    if ($visibleTabs.eq(i).attr('id') === id) {
                        $visibleTabs.eq(i).removeClass('show');
                        $('#footer_nav li').eq(i).removeClass('show');
                        $visibleTabs = $('#canvas_slider > .tab.show');
                        if (!overCurrent) {
                            selectedIndex--;
                            this.update();
                        }
                        ntp.page.footerNavs.visibleTabsChange();
                        ntp.page.sideNavs.visibleTabsChange();
                        break;
                    } else if ($visibleTabs.eq(i).attr('id') === $tab.attr('id')) {
                        overCurrent = true;
                    }
                }
            }
        },
        
        // Selects a tab by its index that is only count for visible tabs.
        selectByIndex: function(index, immediate, forced) {
            var indexChanged = true;
            if (selectedIndex === index) {
                if (!forced) { return; }
                indexChanged = false;
            }
            
            selectedIndex = index;

            // Updates footer navigator class
            $('#footer_nav li.show').each(function (index) {
                if (index + 1 === selectedIndex) {
                    $(this).addClass('highlight');
                } else {
                    $(this).removeClass('highlight');
                }
            });

            // If 'immediate' is auto, then an animation will be used only if canvas is already running an animation.
            if (immediate === 'auto') {
                immediate = !$canvas.is(':animated');
            }

            // Updates tab position
            var $tab = $('.tab.show').eq(index - 1);

            if (tipContentsVisited === true && ntp.content.learnTab.urlIsHilightedContent('tip')) {
                //Leaving learn tab, so unhilight tip
                ntp.content.learnTab.unhilightViewedLearnConents({ tip: $('div#tip_resource_box').data('upload_time') });
                tipContentsVisited = false;

            }else if ($tab.attr('id') === 'tab_learn' && ntp.content.learnTab.urlIsHilightedContent('tip')) {

                //First time Visiting, so notify all others except self.
                ntp.acad.updateLearnContentState('tip', $('div#tip_resource_box').data('upload_time'));
                ntp.acad.updateLearnContentState('tipID', $('div#tip_resource_box').data('tip_ID'));
                ntp.acad.updateLearnContentState('tipExpireDate', $('div#tip_resource_box').data('tip_Expire_Date'));
                tipContentsVisited = true;
            }

            var newScrollLeft = $tab.position().left;
            if ($canvas.scrollLeft() !== newScrollLeft) {
                if (immediate) {
                    $canvas.stop().scrollLeft(newScrollLeft);
                } else {
                    var durationMultiplier = Math.abs(newScrollLeft - $canvas.scrollLeft()) / $canvas.width();
                    if (durationMultiplier > 1) { durationMultiplier = 1; }
                    $canvas.stop().animate(
                        { scrollLeft: newScrollLeft },
                        durationMultiplier * ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION,
                        'linear');
                }
            }
            ntp.page.footerNavs.updateArrow(immediate);
            ntp.page.sideNavs.update(immediate, true, true);
        },
        
        update: function () {
            this.selectByIndex(selectedIndex, 'auto', true);
        },

        // Selects a tab by its id
        selectById: function(id, immediate, forced) {
            var tabIndex = $('.tab.show').index($('#' + id));
            if (tabIndex >= 0 && tabIndex < this.visibleCount()) {
                this.selectByIndex(tabIndex + 1, immediate, forced);
            }
        },
        
        // Auto snaps to the closest tab after scrolling
        snap: function (immediate) {
            var tabWidth = $canvas.width();
            var currentScrollLeft = $canvas.scrollLeft();
            
            // Return if scrolling is caused by resizing
            if (currentScrollLeft === (tabWidth * (selectedIndex - 1))) {
                return;
            }
            
            var closestTabIndex = Math.floor((currentScrollLeft + tabWidth / 2) / tabWidth) + 1;
            this.selectByIndex(closestTabIndex, immediate, true);
        }
    };
})();

/**
 * Handles side navigation arrows
 */
ntp.page.sideNavs = (function () {
    
    var $canvas = null;
    var $tabContents = null;
    var $sideNavs = null;
    
    var tabContentWidth = null;
    var leftSideNavWidth = null;
    var rightSideNavWidth = null;
    
    var overlapped = null;
    
    // Updates the global variable storing left and right of the content of currently selected tab        
    var updateCurrentTabContentWidth = function () {
        if ($tabContents === null) { return null; }
        var selectedIndex = ntp.page.tabs.current();
        var $content = $tabContents.eq(selectedIndex - 1);
        var $columns = $content.children('.column');
        if ($columns.length === 0) {
            tabContentWidth = $content.outerWidth(true);
        } else {
            var $firstColumn = $columns.eq(0);
            var $lastColumn = $columns.eq($columns.length - 1);
            tabContentWidth = $lastColumn.offset().left + $lastColumn.outerWidth(true) - $firstColumn.offset().left;
        }
    };
    
    // Updates side navigator overlap status. If side navigators overlap with page content, they would be hidden.
    var updateOverlap = function (immediate) {
        // Update stored tab content width
        updateCurrentTabContentWidth();

        var tabWidth = $canvas.width();
        var sideNavSpace = (tabWidth - tabContentWidth) / 2;
    
        var newOverlapped = ((sideNavSpace - leftSideNavWidth) < 10 || 
                (sideNavSpace -rightSideNavWidth) < 10);

        if (newOverlapped !== overlapped) {
            if (overlapped === null || immediate) {
                if (newOverlapped) {
                    $sideNavs.addClass('overlapped');
                } else {
                    $sideNavs.removeClass('overlapped');
                }
            } else {
                // Use a fadein or fadeout animation when showing or hiding the side navigators
                $sideNavs.stop(true);
                if (newOverlapped) {
                    $sideNavs.css('opacity', 1);
                    $sideNavs.animate({ opacity: 0 }, 
                        ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2, 
                        'linear', function () {
                        $sideNavs.css('opacity', '');
                        $sideNavs.addClass('overlapped');
                    });
                } else {
                    $sideNavs.css('opacity', 0);
                    $sideNavs.removeClass('overlapped');
                    $sideNavs.animate({ opacity: 1 }, 
                        ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2, 
                        'linear', function () {
                        $sideNavs.css('opacity', '');
                    });
                }
            }

            overlapped = newOverlapped;
        }
    };
    
    // Updates total width of side navigation tools in global variable.
    var updateWidth = function () {
        leftSideNavWidth = 33;
        leftSideNavWidth += $sideNavs.eq(0).outerWidth(true);
        rightSideNavWidth = 33;
        rightSideNavWidth += $sideNavs.eq(1).outerWidth(true);    
    };
    
    // Updates individual side navigator according to currently selected tab.
    var updateNavigator = function (navigatorSelector, navigatorTabIndex, immediate, handleOverlap) {
        var $sideNav = $(navigatorSelector);

        $sideNav.off('click'); // Clear existing event handler

        if (immediate) {
            var sideNavShouldShow = (navigatorTabIndex > 0 && navigatorTabIndex <= ntp.page.tabs.visibleCount()
                                    && ntp.page.tabs.visibleCount() > 1);
            if (sideNavShouldShow) {
                // Target tab should show. 
                var $footerNav = $('#footer_nav li.show').eq(navigatorTabIndex - 1);
                var label = $footerNav.children('.label').text();
                $sideNav.removeClass('disabled')
                        .children('.label')
                        .text(label);

                $sideNav.one('click', function (event) {
                    ntp.page.tabs.selectByIndex(navigatorTabIndex);
                    event.preventDefault();
                });
            } else {
                //Side navigator need to hide.
                $sideNav.addClass('disabled');
            }

            // Update width of side navigators after their size has changed
            updateWidth();

            if (handleOverlap) {
                updateOverlap(true);
            }
        } else {
            // Change navigators with fade in/fade out effect. The implementation is a little bit complicated,
            // as the simple jQuery fadeIn() fadeOut() do not work so well here.
            var fadeIn = function () {
                $sideNav.stop().css('opacity', '');
                updateNavigator(navigatorSelector, navigatorTabIndex, true, handleOverlap);
                if ($sideNav.css('display') !== 'none') {
                    $sideNav.animate({ opacity: 1 }, 
                        ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2, 
                        'linear', function () {
                        $sideNav.css('opacity', '');
                    });
                }
            };

            $sideNav.stop();
            if ($sideNav.css('display') === 'none') {
                setTimeout(fadeIn, ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2);
            } else {
                $sideNav.animate({ opacity: 0 }, 
                    ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2, 
                    'linear', fadeIn);
            }
        }
    };
    
    return {
        init: function () {
            $canvas = $('#canvas');         
            $tabContents = $('#canvas_slider > .tab.show > .content');
            $sideNavs = $('#side_nav li');
        },
        
        visibleTabsChange: function () {
            $tabContents = $('#canvas_slider > .tab.show > .content');
            this.update(true, true, true);
        },
        
        // Updates both side navigators according to currently selected tab.
        update: function (immediate, handleOverlap, forced) {
            var currentTab = ntp.page.tabs.current();
            
            if (forced) {
                // Update side navigators
                updateNavigator('#left_side_nav', currentTab - 1, immediate, handleOverlap);
                updateNavigator('#right_side_nav', currentTab + 1, immediate, handleOverlap);
            } else {
                updateOverlap();
            }
        }
    };
})();

/**
 * Handles footer navigation buttons
 */
ntp.page.footerNavs = (function () {
    var $canvas = null;
    var $footerNavs = null;
    var $navArrow = null;
    var $navRoot = null;
    var footerNavWidth = null;
    var isInit = false;
    
    return {
        init: function () {
            $canvas = $('#canvas');
            $navArrow = $('#nav_arrow');
            $footerNavs = $('#footer_nav li.show');
            $navRoot = $('#footer_nav');
            if(!isInit){
                $('#footer_nav li .button').each(
                        function(index) {
                            var tabId = $('#canvas_slider > .tab').eq(index).attr('id');
                            $(this).on("click", function() {
                                ntp.page.tabs.selectById(tabId);
                                event.preventDefault();
                            });
                        });
            }
            isInit = true;
            if($footerNavs.length > 1)
                this.visibleTabsChange();
        },
        
        visibleTabsChange: function () {
            $footerNavs = $('#footer_nav li.show');
            // Makes the footer navs equal width, and store footerNavWidth
            footerNavWidth = 0;

            if ($footerNavs.length > 1) {
                var innerWidth = 0;
                $navRoot.removeClass('hidden');
                $footerNavs.each(function() {
                    $(this).width('auto');
                    var currentNavWidth = $(this).width();
                    if (currentNavWidth > innerWidth) {
                        innerWidth = currentNavWidth;
                    }
                });

                $footerNavs.each(function(index) {
                    if (index + 1 === ntp.page.tabs.current()) {
                        $(this).addClass('highlight');
                    } else {
                        $(this).removeClass('highlight');
                    }
                    $(this).width(innerWidth);
                });

                // Calculate width from offset of two elements. This is needed to count in a mysterious space between the elements in the width
                // (elements already have margin = 0 )
                var firstTabLeft = $footerNavs.eq(0).offset().left;
                var secondTabLeft = $footerNavs.eq(1).offset().left;
                if (firstTabLeft === secondTabLeft && firstTabLeft > 0)
                    footerNavWidth = $footerNavs.eq(0).innerWidth();
                else
                    footerNavWidth = $footerNavs.eq(1).offset().left - $footerNavs.eq(0).offset().left;
                $navArrow.removeClass("hidden");
                this.updateArrow(true);
            } else {
                $navRoot.addClass('hidden');
                $navArrow.addClass("hidden");
            }
        },
        
        updateArrow: function(immediate){
            if(!$('#nav_arrow').hasClass("hidden")){
                var index = ntp.page.tabs.current() - 1;
                var footerFirstTab = $footerNavs.eq(0);
                var footerLeft = footerFirstTab.offset().left;
                var newOffset = footerLeft + footerNavWidth * (ntp.page.tabs.current() - 0.5) - $navArrow.width() / 2;
                if(immediate){
                    $navArrow.stop().css('left', newOffset);
                } else {
                    var oldLeft = $navArrow.css('left');
                    if(oldLeft !== newOffset){
                        $navArrow.stop().animate(
                                {left: newOffset},
                                ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION,
                                'linear');
                    }
                }
            }
        },
        
        update: function () {
            if (!footerNavWidth) { // footerNavWidth might be 0 if the window size is too small
                this.visibleTabsChange();
            }
            this.updateArrow(true);
        }
    };
})();


/**
 * Page initialization on customization complete
 */
ntp.deferred.customization.done(function () {
    ntp.page.layout.init();
    ntp.page.tabs.init();
    ntp.widget.templatebar.init();
    
    if (ntp.page.tabs.count() > 1) {
        ntp.page.sideNavs.init();
    }

    $(window).resize(function() {
        ntp.widget.templatebar.update();
        ntp.widget.projectbar.update();
        ntp.page.layout.update();

        if (ntp.page.tabs.count() > 1) {
            ntp.page.footerNavs.update();
            // Fix the problem of window resizing when switch tab transition is not finished.
            ntp.page.tabs.update();
        }
    });

    $(document).on('keydown', function(e) {
        if (e.ctrlKey || e.altKey) {
            ntp.acad.notifyShortcutKeyPressed(e);
        }
    });
    
    $(document).mousedown(function(e) {
        if(e.button == 1) {
            e.preventDefault();
        }
    });

    // Initialize layouts
    ntp.page.layout.update();
    
    // Show create tab on start
    if (ntp.page.tabs.count() > 1) {
        ntp.page.tabs.selectById('tab_create', true, true);
    }
});

/**
 * Page initialization on customization and localization complete.
 */
$.when(ntp.deferred.localization, ntp.deferred.customization).done(function () {
    // Updates layout after text is loaded. Some elements are resized automatically.
    if (ntp.page.tabs.count() > 1) {
        ntp.page.footerNavs.init();
        ntp.page.sideNavs.update(true, true, true); // Update side navigator names
    }
});

/**
 * Allows only dropping dwg files
 */
(function () {
    function isDrawingFile(path) {
        path = path || "";
        path = path.substr(path.length - 4).toLowerCase();
        return (path == ".dxf") || (path == ".dwg") || (path == ".dws") || (path == ".dwt");
    }

    function queryDropHandler(e) {
        e.preventDefault();
        e.stopPropagation();

        var data = e.originalEvent.dataTransfer;
        data.dropEffect = "none";
        if (data && data.files) {
            for (var i = 0; i < data.files.length; ++i) {
                if (isDrawingFile(data.files[i].name)) {
                    data.dropEffect = "copy";
                    break;
                }
            }
        }
    }

    $(window).on('dragenter', queryDropHandler);
    $(window).on('dragover', queryDropHandler);
    $(window).on('drop', function(e) {
        e.preventDefault();
        e.stopPropagation();
        ntp.acad.openDroppedFiles();
    });
})();

// SIG // Begin signature block
// SIG // MIIZ7QYJKoZIhvcNAQcCoIIZ3jCCGdoCAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // /snx1KQ/fWt3XlbDkmog7kHZ8S1RiWwaq/h8TxM3iLug
// SIG // ggpjMIIFAjCCA+qgAwIBAgIQZLyxx87k2lhp1Y8eIvvp
// SIG // 9DANBgkqhkiG9w0BAQsFADB/MQswCQYDVQQGEwJVUzEd
// SIG // MBsGA1UEChMUU3ltYW50ZWMgQ29ycG9yYXRpb24xHzAd
// SIG // BgNVBAsTFlN5bWFudGVjIFRydXN0IE5ldHdvcmsxMDAu
// SIG // BgNVBAMTJ1N5bWFudGVjIENsYXNzIDMgU0hBMjU2IENv
// SIG // ZGUgU2lnbmluZyBDQTAeFw0xNTA5MDMwMDAwMDBaFw0x
// SIG // NjA5MDIyMzU5NTlaMIGIMQswCQYDVQQGEwJVUzETMBEG
// SIG // A1UECAwKQ2FsaWZvcm5pYTETMBEGA1UEBwwKU2FuIFJh
// SIG // ZmFlbDEWMBQGA1UECgwNQXV0b2Rlc2ssIEluYzEfMB0G
// SIG // A1UECwwWRGVzaWduIFNvbHV0aW9ucyBHcm91cDEWMBQG
// SIG // A1UEAwwNQXV0b2Rlc2ssIEluYzCCASIwDQYJKoZIhvcN
// SIG // AQEBBQADggEPADCCAQoCggEBAOqZ9OjPzARqd9dP4fq2
// SIG // qXdpTJolFYJ8Wxg8WcZVciX3l+dljew4R9C8WR/0tW17
// SIG // lLANaSORF4IYmWTgdpFg5PGR3s+R4A461hpRIG9QthGU
// SIG // q8Bt7af5VbCjjKtYBNq/x3ukqVsw/1/qljGVplDOyr97
// SIG // 6ktnCm0/wL0N+ubk5WAIgv+I9E0i6+GX1hRm+eRliYKe
// SIG // KF0/gBLyDh3ut+N3HNPrnfjfec7q2Czd82CeRBCToR1E
// SIG // 7Cp6yBBdGUfb3JE0PnE1+6n0qJnsvs2bsCIzqVo3A0iM
// SIG // HBsbioxa5DYyXuEfNCktosUlFXfegd514ZJns6YDr4Pl
// SIG // ahC0+lK3hgJy5vkCAwEAAaOCAW4wggFqMAkGA1UdEwQC
// SIG // MAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsG
// SIG // AQUFBwMDMGYGA1UdIARfMF0wWwYLYIZIAYb4RQEHFwMw
// SIG // TDAjBggrBgEFBQcCARYXaHR0cHM6Ly9kLnN5bWNiLmNv
// SIG // bS9jcHMwJQYIKwYBBQUHAgIwGRoXaHR0cHM6Ly9kLnN5
// SIG // bWNiLmNvbS9ycGEwHwYDVR0jBBgwFoAUljtT8Hkzl699
// SIG // g+8uK8zKt4YecmYwKwYDVR0fBCQwIjAgoB6gHIYaaHR0
// SIG // cDovL3N2LnN5bWNiLmNvbS9zdi5jcmwwVwYIKwYBBQUH
// SIG // AQEESzBJMB8GCCsGAQUFBzABhhNodHRwOi8vc3Yuc3lt
// SIG // Y2QuY29tMCYGCCsGAQUFBzAChhpodHRwOi8vc3Yuc3lt
// SIG // Y2IuY29tL3N2LmNydDARBglghkgBhvhCAQEEBAMCBBAw
// SIG // FgYKKwYBBAGCNwIBGwQIMAYBAQABAf8wDQYJKoZIhvcN
// SIG // AQELBQADggEBAB6BYdY8nzLWS3nsk/xM5CWdizR4wyDn
// SIG // 8wfGVbcuEq/88byNnY0ODxisa7AELvLe6b48DorFFP2n
// SIG // Hw0qOeIxfg2w+1DzWzYQhMfLi4V/Pp1OmCYvYEsTb/oU
// SIG // +YbiSgSFjrN5Pi2LN5Oxw5Ey7WAK2vhWr4EL/7QDHoso
// SIG // 4a3lS3WklPz2K/Nlhk4i8S230BlGCHsZGDY7PtfowWdz
// SIG // cq50xCEIScERSRtLd+YL6SWXk+6gX9HVcFxtvLQ4N9Df
// SIG // F/BKk13pYgyvZPkedIKA4yYCy4/VjouNoKhgM/aaGLAy
// SIG // Dv3ujRdXLCcosR4MbD8X1v9z6WkgG5YZZicWPPsJN7ty
// SIG // B1wwggVZMIIEQaADAgECAhA9eNf5dklgsmF99PAeyoYq
// SIG // MA0GCSqGSIb3DQEBCwUAMIHKMQswCQYDVQQGEwJVUzEX
// SIG // MBUGA1UEChMOVmVyaVNpZ24sIEluYy4xHzAdBgNVBAsT
// SIG // FlZlcmlTaWduIFRydXN0IE5ldHdvcmsxOjA4BgNVBAsT
// SIG // MShjKSAyMDA2IFZlcmlTaWduLCBJbmMuIC0gRm9yIGF1
// SIG // dGhvcml6ZWQgdXNlIG9ubHkxRTBDBgNVBAMTPFZlcmlT
// SIG // aWduIENsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlm
// SIG // aWNhdGlvbiBBdXRob3JpdHkgLSBHNTAeFw0xMzEyMTAw
// SIG // MDAwMDBaFw0yMzEyMDkyMzU5NTlaMH8xCzAJBgNVBAYT
// SIG // AlVTMR0wGwYDVQQKExRTeW1hbnRlYyBDb3Jwb3JhdGlv
// SIG // bjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29y
// SIG // azEwMC4GA1UEAxMnU3ltYW50ZWMgQ2xhc3MgMyBTSEEy
// SIG // NTYgQ29kZSBTaWduaW5nIENBMIIBIjANBgkqhkiG9w0B
// SIG // AQEFAAOCAQ8AMIIBCgKCAQEAl4MeABavLLHSCMTXaJNR
// SIG // YB5x9uJHtNtYTSNiarS/WhtR96MNGHdou9g2qy8hUNqe
// SIG // 8+dfJ04LwpfICXCTqdpcDU6kDZGgtOwUzpFyVC7Oo9tE
// SIG // 6VIbP0E8ykrkqsDoOatTzCHQzM9/m+bCzFhqghXuPTbP
// SIG // HMWXBySO8Xu+MS09bty1mUKfS2GVXxxw7hd924vlYYl4
// SIG // x2gbrxF4GpiuxFVHU9mzMtahDkZAxZeSitFTp5lbhTVX
// SIG // 0+qTYmEgCscwdyQRTWKDtrp7aIIx7mXK3/nVjbI13Iwr
// SIG // b2pyXGCEnPIMlF7AVlIASMzT+KV93i/XE+Q4qITVRrgT
// SIG // hsIbnepaON2b2wIDAQABo4IBgzCCAX8wLwYIKwYBBQUH
// SIG // AQEEIzAhMB8GCCsGAQUFBzABhhNodHRwOi8vczIuc3lt
// SIG // Y2IuY29tMBIGA1UdEwEB/wQIMAYBAf8CAQAwbAYDVR0g
// SIG // BGUwYzBhBgtghkgBhvhFAQcXAzBSMCYGCCsGAQUFBwIB
// SIG // FhpodHRwOi8vd3d3LnN5bWF1dGguY29tL2NwczAoBggr
// SIG // BgEFBQcCAjAcGhpodHRwOi8vd3d3LnN5bWF1dGguY29t
// SIG // L3JwYTAwBgNVHR8EKTAnMCWgI6Ahhh9odHRwOi8vczEu
// SIG // c3ltY2IuY29tL3BjYTMtZzUuY3JsMB0GA1UdJQQWMBQG
// SIG // CCsGAQUFBwMCBggrBgEFBQcDAzAOBgNVHQ8BAf8EBAMC
// SIG // AQYwKQYDVR0RBCIwIKQeMBwxGjAYBgNVBAMTEVN5bWFu
// SIG // dGVjUEtJLTEtNTY3MB0GA1UdDgQWBBSWO1PweTOXr32D
// SIG // 7y4rzMq3hh5yZjAfBgNVHSMEGDAWgBR/02Wnwt3su/Aw
// SIG // CfNDOfoCrzMxMzANBgkqhkiG9w0BAQsFAAOCAQEAE4Ua
// SIG // HmmpN/egvaSvfh1hU/6djF4MpnUeeBcj3f3sGgNVOftx
// SIG // lcdlWqeOMNJEWmHbcG/aIQXCLnO6SfHRk/5dyc1eA+CJ
// SIG // nj90Htf3OIup1s+7NS8zWKiSVtHITTuC5nmEFvwosLFH
// SIG // 8x2iPu6H2aZ/pFalP62ELinefLyoqqM9BAHqupOiDlAi
// SIG // KRdMh+Q6EV/WpCWJmwVrL7TJAUwnewusGQUioGAVP9rJ
// SIG // +01Mj/tyZ3f9J5THujUOiEn+jf0or0oSvQ2zlwXeRAwV
// SIG // +jYrA9zBUAHxoRFdFOXivSdLVL4rhF4PpsN0BQrvl8OJ
// SIG // IrEfd/O9zUPU8UypP7WLhK9k8tAUITGCDuIwgg7eAgEB
// SIG // MIGTMH8xCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1h
// SIG // bnRlYyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50
// SIG // ZWMgVHJ1c3QgTmV0d29yazEwMC4GA1UEAxMnU3ltYW50
// SIG // ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENB
// SIG // AhBkvLHHzuTaWGnVjx4i++n0MA0GCWCGSAFlAwQCAQUA
// SIG // oHwwEAYKKwYBBAGCNwIBDDECMAAwGQYJKoZIhvcNAQkD
// SIG // MQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwG
// SIG // CisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEILsLrCT6
// SIG // veFmuzAPN8+CeMYdXbYAV3CEEaw4emc40hHLMA0GCSqG
// SIG // SIb3DQEBAQUABIIBADB37v55MYbzQFXjYMgr2T5hXeC/
// SIG // 5lY5C06QB87vhXeaLqBHIY5d2S2KBpbQT+1a2Kr6vSiX
// SIG // MyNu0tZ/rZb8vysUpyes4JV36ZYde1O6TfwfAM1nCMf+
// SIG // NSfp++gQyV97wqOBb4QZJ/ZT9C5X9DEe4OmZTLOtaQWE
// SIG // pCFvb06q6wDHxZbYtJlQGtp+Vfeaf9eSKt1UAMOh56C5
// SIG // 0B8jX855NwIouZk0nx0MAcz/HgbNyZ64AhlVyN4SWXZy
// SIG // 3w+s0tNrqjGLS2vya/rXzNGZ9+Ljc2/O4QxrFSKdcXxk
// SIG // PaNiUfqa1vT0SIBLeD4J4hVgYLbOZqxzdWw1kxiAXzeg
// SIG // w3wW10ahggyhMIIMnQYKKwYBBAGCNwMDATGCDI0wggyJ
// SIG // BgkqhkiG9w0BBwKgggx6MIIMdgIBAzEPMA0GCWCGSAFl
// SIG // AwQCAQUAMIHdBgsqhkiG9w0BCRABBKCBzQSByjCBxwIB
// SIG // AQYJKwYBBAGgMgIDMDEwDQYJYIZIAWUDBAIBBQAEIDZq
// SIG // wFM2YY7bBEIGT4IMIraNQ/ZONYKmnkVIVFChDVvnAhQP
// SIG // Q2Q7ddte4/hp48HfJJ155vl1gBgPMjAxNjAyMDcwNTA5
// SIG // MTVaoF2kWzBZMQswCQYDVQQGEwJTRzEfMB0GA1UEChMW
// SIG // R01PIEdsb2JhbFNpZ24gUHRlIEx0ZDEpMCcGA1UEAxMg
// SIG // R2xvYmFsU2lnbiBUU0EgZm9yIEFkdmFuY2VkIC0gRzKg
// SIG // ggjGMIIEqTCCA5GgAwIBAgISESEWwAmY3MaPon0lw4Y2
// SIG // qIO7MA0GCSqGSIb3DQEBCwUAMFsxCzAJBgNVBAYTAkJF
// SIG // MRkwFwYDVQQKExBHbG9iYWxTaWduIG52LXNhMTEwLwYD
// SIG // VQQDEyhHbG9iYWxTaWduIFRpbWVzdGFtcGluZyBDQSAt
// SIG // IFNIQTI1NiAtIEcyMB4XDTE1MDIwMzAwMDAwMFoXDTI2
// SIG // MDMwMzAwMDAwMFowWTELMAkGA1UEBhMCU0cxHzAdBgNV
// SIG // BAoTFkdNTyBHbG9iYWxTaWduIFB0ZSBMdGQxKTAnBgNV
// SIG // BAMTIEdsb2JhbFNpZ24gVFNBIGZvciBBZHZhbmNlZCAt
// SIG // IEcyMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKC
// SIG // AQEAt8eSiijY85HLUA1d6ylEwYJZpcpkzr2DtVHllMxu
// SIG // gJoXbNz/YuY2Q14Bwgxkjcp3nvZdd0W5Xsgk60E3EWY3
// SIG // rMjNdJvMs+9M4n8bznO+yZadNR5891T3bOygEtaumZ1j
// SIG // xHhfWakuIvXDJ9iyW0VjJ5D/jLE9qbTzDsDxIGUGiZ0D
// SIG // HKOwUkGFNOu8y85U9hL+swGJSqb6VMUf+9McwOO/l+M/
// SIG // 0Bv5QMcb1esmfEMwx3DyeBGqhU6lDg0ASjoRd6PKIwGA
// SIG // nsiTVl1GzIrY0qoKhIWyiVvVMs5BexhOVMqr0VJeEtMk
// SIG // nlbzDI/dvZKOedVBicOjoOWR/MdnnYczznEycwIDAQAB
// SIG // o4IBZzCCAWMwDgYDVR0PAQH/BAQDAgeAMEwGA1UdIARF
// SIG // MEMwQQYJKwYBBAGgMgEeMDQwMgYIKwYBBQUHAgEWJmh0
// SIG // dHBzOi8vd3d3Lmdsb2JhbHNpZ24uY29tL3JlcG9zaXRv
// SIG // cnkvMAkGA1UdEwQCMAAwFgYDVR0lAQH/BAwwCgYIKwYB
// SIG // BQUHAwgwRgYDVR0fBD8wPTA7oDmgN4Y1aHR0cDovL2Ny
// SIG // bC5nbG9iYWxzaWduLmNvbS9ncy9nc3RpbWVzdGFtcGlu
// SIG // Z3NoYTJnMi5jcmwwWAYIKwYBBQUHAQEETDBKMEgGCCsG
// SIG // AQUFBzAChjxodHRwOi8vc2VjdXJlLmdsb2JhbHNpZ24u
// SIG // Y29tL2NhY2VydC9nc3RpbWVzdGFtcGluZ3NoYTJnMi5j
// SIG // cnQwHQYDVR0OBBYEFC1ubtGN5QOA7udj6afZ2gs8VyI9
// SIG // MB8GA1UdIwQYMBaAFJIhp0qVXWSwm7Qe5gA3R+adQStM
// SIG // MA0GCSqGSIb3DQEBCwUAA4IBAQB/CgOhVXb+qvoU0RF6
// SIG // 4uIPOjL/e0bGBHGMXtDMeUfddQdtuki+P2DiyxEgwrar
// SIG // Q8e3otm4Wv7NtDqz8VJTDLSO/2LdiJuyGn8bUbdxh9pD
// SIG // SzdSzf4+BilwpCds59w8f3bjkMga6ZkSUO7qTCMayF+c
// SIG // 1SmEKuaBYgZ2gnajE1UOSKZR0fNMDTwIICiGaoO3nQyg
// SIG // XmA0BjGofXGPUfIE36Imk7nVKHejoLdCo4swJ1oJ9tyD
// SIG // mCHVfUGaBxkUWih2fa2goyQBTuEWyL9/R/M6BF5lPqJM
// SIG // //1vBs+suBsAZ/rva37X4gq/e8WnEqzXNPVXiD6Elq1r
// SIG // VC49Jw/iORrBxxhyMIIEFTCCAv2gAwIBAgILBAAAAAAB
// SIG // MYnGUAQwDQYJKoZIhvcNAQELBQAwTDEgMB4GA1UECxMX
// SIG // R2xvYmFsU2lnbiBSb290IENBIC0gUjMxEzARBgNVBAoT
// SIG // Ckdsb2JhbFNpZ24xEzARBgNVBAMTCkdsb2JhbFNpZ24w
// SIG // HhcNMTEwODAyMTAwMDAwWhcNMjkwMzI5MTAwMDAwWjBb
// SIG // MQswCQYDVQQGEwJCRTEZMBcGA1UEChMQR2xvYmFsU2ln
// SIG // biBudi1zYTExMC8GA1UEAxMoR2xvYmFsU2lnbiBUaW1l
// SIG // c3RhbXBpbmcgQ0EgLSBTSEEyNTYgLSBHMjCCASIwDQYJ
// SIG // KoZIhvcNAQEBBQADggEPADCCAQoCggEBAKqbjsOrEVEl
// SIG // AbaWlOJP2MEI9kYj2UXFlZdbqxq/0mxXyTMGH6APxjx+
// SIG // U0h6v52Hnq/uw4xH4ULs4+OhSmwMF8SmwbnNW/EeRImO
// SIG // /gveIVgT7k3IxWcLHLKz8TR2kaLLB203xaBHJgIVpJCR
// SIG // qXme1+tXnSt8ItgU1/EHHngiNmt3ea+v+X+OTuG1CDH9
// SIG // 6u1LcWKMI/EDOY9EebZ2A1eerS8IRtzSjLz0jnTOyGhp
// SIG // UXYRiw9dJFsZVD0mzECNgicbWSB9WfaTgI74Kjj9a6BA
// SIG // ZR9XdsxbjgRPLKjbhFATT8bci7n43WlMiOucezAm/HpY
// SIG // u1m8FHKSgVe3dsnYgAqAbgkCAwEAAaOB6DCB5TAOBgNV
// SIG // HQ8BAf8EBAMCAQYwEgYDVR0TAQH/BAgwBgEB/wIBADAd
// SIG // BgNVHQ4EFgQUkiGnSpVdZLCbtB7mADdH5p1BK0wwRwYD
// SIG // VR0gBEAwPjA8BgRVHSAAMDQwMgYIKwYBBQUHAgEWJmh0
// SIG // dHBzOi8vd3d3Lmdsb2JhbHNpZ24uY29tL3JlcG9zaXRv
// SIG // cnkvMDYGA1UdHwQvMC0wK6ApoCeGJWh0dHA6Ly9jcmwu
// SIG // Z2xvYmFsc2lnbi5uZXQvcm9vdC1yMy5jcmwwHwYDVR0j
// SIG // BBgwFoAUj/BLf6guRSSuTVD6Y5qL3uLdG7wwDQYJKoZI
// SIG // hvcNAQELBQADggEBAARWgkp80M7JvzZm0b41npNsl+gG
// SIG // zjEYWflsQV+ALsBCJbgYx/zUsTfEaKDPKGoDdEtjl4V3
// SIG // YTvXL+P1vTOikn0RH56KbO8ssPRijTZz0RY28bxe7LSA
// SIG // mHj80nZ56OEhlOAfxKLhqmfbs5xz5UAizznO2+Z3lae7
// SIG // ssv2GYadn8jUmAWycW9Oda7xPWRqO15ORqYqXQiS8aPz
// SIG // HXS/Yg0jjFwqOJXSwNXNz4jaHyi1uoFpZCq1pqLVc6/c
// SIG // RtsErpHXbsWYutRHxFZ0gEd4WIy+7yv97Gy/0ZT3v1Dg
// SIG // e+CQ/SAYeBgiXQgujBygl/MdmX2jnZHTBkROBG56HCDj
// SIG // NvC2ULkxggK0MIICsAIBATBxMFsxCzAJBgNVBAYTAkJF
// SIG // MRkwFwYDVQQKExBHbG9iYWxTaWduIG52LXNhMTEwLwYD
// SIG // VQQDEyhHbG9iYWxTaWduIFRpbWVzdGFtcGluZyBDQSAt
// SIG // IFNIQTI1NiAtIEcyAhIRIRbACZjcxo+ifSXDhjaog7sw
// SIG // DQYJYIZIAWUDBAIBBQCgggEUMBoGCSqGSIb3DQEJAzEN
// SIG // BgsqhkiG9w0BCRABBDAcBgkqhkiG9w0BCQUxDxcNMTYw
// SIG // MjA3MDUwOTE1WjAvBgkqhkiG9w0BCQQxIgQgokfqpEVh
// SIG // xHWTXL3yKUDYDRU23r17MlvinFPW0Uvh4kEwgaYGCyqG
// SIG // SIb3DQEJEAIMMYGWMIGTMIGQMIGNBBRXqt6jTjqEJxGX
// SIG // sll4jXMMauIuyTB1MF+kXTBbMQswCQYDVQQGEwJCRTEZ
// SIG // MBcGA1UEChMQR2xvYmFsU2lnbiBudi1zYTExMC8GA1UE
// SIG // AxMoR2xvYmFsU2lnbiBUaW1lc3RhbXBpbmcgQ0EgLSBT
// SIG // SEEyNTYgLSBHMgISESEWwAmY3MaPon0lw4Y2qIO7MA0G
// SIG // CSqGSIb3DQEBAQUABIIBADd7U+/r4Qbl/wkb20/KO/pN
// SIG // TQPFapQUUgYWLhIwAlqMNky9bcPbGD77oMQnWCsuLuhK
// SIG // XmZeVhd1AxWAZpi3OePOYrGalJwGRVBz74EgbqsMVCbi
// SIG // P3jLcrQfEZHAupLRTv52PDHNP56eGDK/eG1Y+CKt9Rcl
// SIG // M08Hn5VO0A/SC9L9/J+W9XAsRx4WIEZHKbUp4kipbfpP
// SIG // qofigcYcDu/XW9OguQI5O/UB4W3rhb3yq6UbZfyhMysb
// SIG // Jp7Naq+ZE5r+K9e/MUTe1LaRrOY8vout1ED44vvikCQX
// SIG // btMAPgTiucSZjvxnLAjg4zDFFRSC05Er1iQ/JaqRczyN
// SIG // jMDHEJ2eJxY=
// SIG // End signature block
